def pivot(tableau, pivot_row, pivot_col):
    """
    Performs a pivot operation on the tableau.
    """
    m, n = len(tableau), len(tableau[0])
    pivot_element = tableau[pivot_row][pivot_col]

    # Normalize the pivot row.
    for j in range(n):
        tableau[pivot_row][j] /= pivot_element

    # Update the rest of the tableau.
    for i in range(m):
        if i != pivot_row:
            factor = tableau[i][pivot_col]
            for j in range(n):
                tableau[i][j] -= factor * tableau[pivot_row][j]

def simplex_method(A, b, c, tol=1e-9):
    """
    Solves the LP:
        maximize    c[0]*x + c[1]*y
        subject to  A[i][0]*x + A[i][1]*y <= b[i] for all i,
                    x, y >= 0.
    The method uses the simplex algorithm.
    
    Parameters:
        A   : List of lists. Each inner list contains coefficients for x and y.
        b   : List of right-hand side values.
        c   : List or tuple with 2 elements representing the objective coefficients.
        tol : Tolerance for numerical comparisons.
    
    Returns:
        A tuple (opt_value, solution) where:
            opt_value: the maximum objective value.
            solution: a list with the optimal values for [x, y].
            
    Raises:
        ValueError: if the problem is unbounded.
    """
    m = len(A)      # number of constraints
    n = 2           # number of original variables (x and y)
    
    # Total variables = original variables + slack variables
    total_vars = n + m

    # Build the initial tableau.
    # Each constraint i becomes: A[i][0]*x + A[i][1]*y + s_i = b[i]
    # Tableau has m rows (for constraints) and one row for objective.
    # Columns: [x, y, slack variables..., RHS]
    tableau = []
    for i in range(m):
        row = [0.0] * (total_vars + 1)
        row[0] = A[i][0]
        row[1] = A[i][1]
        # slack variable coefficient (1 for its own constraint)
        row[n + i] = 1.0
        # right-hand side
        row[-1] = b[i]
        tableau.append(row)
    
    # Objective row: maximize c[0]*x + c[1]*y  => in tableau, we use -c for the objective row.
    obj_row = [0.0] * (total_vars + 1)
    obj_row[0] = -c[0]
    obj_row[1] = -c[1]
    # RHS of objective row is 0
    tableau.append(obj_row)

    m_rows = m      # number of constraint rows
    n_cols = total_vars + 1  # total columns in tableau

    # List of basic variable indices for each row.
    # Initially, the slack variables are basic.
    basic_vars = [n + i for i in range(m)]
    
    while True:
        # Find the entering variable (most negative coefficient in the objective row)
        pivot_col = None
        last_row = tableau[-1]
        min_value = tol
        for j in range(total_vars):
            if last_row[j] < -tol and last_row[j] < min_value:
                min_value = last_row[j]
                pivot_col = j

        # If no negative coefficient, optimal solution found.
        if pivot_col is None:
            break

        # Determine the leaving variable by minimum ratio test.
        pivot_row = None
        min_ratio = float("inf")
        for i in range(m_rows):
            if tableau[i][pivot_col] > tol:
                ratio = tableau[i][-1] / tableau[i][pivot_col]
                if ratio < min_ratio:
                    min_ratio = ratio
                    pivot_row = i

        if pivot_row is None:
            raise ValueError("The linear program is unbounded.")

        # Pivot on the identified pivot element.
        basic_vars[pivot_row] = pivot_col

    # Extract the solution for the original variables (x and y).
    solution = [0.0] * n
    for i in range(m_rows):
        if basic_vars[i] < n:
            solution[basic_vars[i]] = tableau[i][-1]

    opt_value = tableau[-1][-1]
    # Since we used -c in the objective row, the optimal value is -opt_value.
    return opt_value, solution

if __name__ == "__main__":
    c = list(map(float, input().split()))
    n = int(input())
    b_vec = list(map(float, input().split()))
    A = [list(map(float, input().split())) for _ in range(n)]
    indices = list(map(int, input().split()))
    i_idx, j_idx = indices[0] - 1, indices[1] - 1

    answer, _ = simplex_method(A, b_vec, c)
    print(f'{round(answer, 5)}')